import paddle
import numpy as np
from paddle_quantum.circuit import UAnsatz
paddle.set_default_dtype("float64")


# Construct parameterized quantum circuit YZY
def U_cir(train_block, w_theta, x):
    cir = UAnsatz(1)
    for i in range(train_block):
        cir.ry(w_theta[i], 0)
        cir.rz(x, 0)  # input data
    cir.ry(w_theta[-1], 0)
    return cir


# Create model
class QNN(paddle.nn.Layer):
    def __init__(self, 
                 train_block,           # L layer
                 SEED=0,
                 dtype='float64'):
        super(QNN, self).__init__()
        self.train_block = train_block
        paddle.seed(SEED)
        # initiate trainable parameter 
        self.w_theta = self.create_parameter(
            shape=[(train_block+1)],
            default_initializer=paddle.nn.initializer.Uniform(0.0, 2*np.pi),
            dtype=dtype,
            is_bias=False)


    def forward(self, x):
        """
        Forward propagation
        """
        predict = []
        H_info = [[1.0, 'z0']]
        x = paddle.to_tensor(x, dtype='float64')
        if len(x.shape) == 1:  # 1-dimension data
            x = x.reshape((-1, 1))
        for i in range(x.shape[0]):
            cir = U_cir(self.train_block, self.w_theta, x[i])
            # Run the quantum circuit
            cir.run_state_vector()
            predict.append(cir.expecval(H_info))
        return paddle.concat(predict).reshape((-1,)), cir


# Training
def train_qnn(x, y, train_block, LR, ITR, SEED, BATCHSIZE=20):
    model = QNN(train_block, SEED)
    opt = paddle.optimizer.Adam(learning_rate=LR, parameters=model.parameters())
    loss_list = []
    x = paddle.to_tensor(x, dtype='float64')
    y = paddle.to_tensor(y, dtype='float64')
    for ep in range(1, ITR + 1):
        # Select batch of data
        for itr in range(len(x) // BATCHSIZE):
            x_batch = x[itr * BATCHSIZE:(itr + 1) * BATCHSIZE]
            y_batch = y[itr * BATCHSIZE:(itr + 1) * BATCHSIZE]
            # Run the network defined above
            predict, cir = model(x_batch)
            avg_loss = paddle.mean((predict - y_batch) ** 2)
            loss_list.append(avg_loss.numpy())
            # Calculate the gradient and optimize
            avg_loss.backward()
            opt.minimize(avg_loss)
            opt.clear_grad()
            if (itr+1) % 5 == 0:
                print("qnn:epoch:", ep,"qnn:iter:", (itr+1), " train  loss:", "%.8f" % avg_loss.numpy())

    return model, loss_list
